<?php
// ═══════════════════════════════════════════════════════════════
//  EKLESIA STANDALONE — config.php
//  Versão autónoma para instalação em domínio próprio
// ═══════════════════════════════════════════════════════════════

define('EKLESIA_VERSION',   '1.0.0');
define('EKLESIA_HOME',      'https://eklesia.asaf.fyi');       // servidor principal
define('EKLESIA_ACTIVATE',  'https://eklesia.asaf.fyi/api/activate.php');
define('EKLESIA_VERIFY',    'https://eklesia.asaf.fyi/api/verify.php');
define('CONTACT_EMAIL',     'audivido3@gmail.com');

define('DATA_DIR',      __DIR__ . '/../data');
define('CHURCHES_DIR',  DATA_DIR . '/churches');
define('SYSTEM_FILE',   DATA_DIR . '/system.json');
define('LICENSE_FILE',  DATA_DIR . '/license.json');

// Detectar URL base desta instalação
$_protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
$_rawHost  = strtolower($_SERVER['HTTP_HOST'] ?? 'localhost');
// Normalizar: remover www. e porta (igual ao servidor de activação)
$_host     = preg_replace('/^www\./', '', $rawHost ?? $_rawHost);
$_host     = preg_replace('/:\d+$/', '', $_host);
define('INSTALL_DOMAIN', $_host);
define('INSTALL_URL',    $_protocol . '://' . ($_SERVER['HTTP_HOST'] ?? 'localhost'));

if (session_status() === PHP_SESSION_NONE) {
    ini_set('session.cookie_httponly', 1);
    ini_set('session.use_strict_mode', 1);
    session_set_cookie_params(['samesite' => 'Strict']);
    session_start();
}

foreach ([DATA_DIR, CHURCHES_DIR] as $dir) {
    if (!is_dir($dir)) mkdir($dir, 0755, true);
}

// Proteger pasta data
if (!file_exists(DATA_DIR . '/.htaccess')) {
    file_put_contents(DATA_DIR . '/.htaccess', "Deny from all\n");
}

require_once __DIR__ . '/helpers.php';
require_once __DIR__ . '/data.php';
require_once __DIR__ . '/lang.php';

// ─── VERIFICAR ACTIVAÇÃO ─────────────────────────────────────
function isActivated(): bool {
    if (!file_exists(LICENSE_FILE)) return false;
    $lic = json_decode(file_get_contents(LICENSE_FILE), true);
    if (!$lic || empty($lic['activated'])) return false;
    // Verificar que o domínio bate certo
    if (($lic['domain'] ?? '') !== INSTALL_DOMAIN) return false;
    // Re-verificar online a cada 7 dias
    $lastVerify = $lic['last_verify'] ?? 0;
    if ((time() - $lastVerify) > 86400 * 7) {
        return verifyOnline($lic['key'] ?? '');
    }
    return true;
}

function verifyOnline(string $key): bool {
    if (empty($key)) return false;
    $ctx = stream_context_create(['http' => [
        'method'  => 'POST',
        'header'  => 'Content-Type: application/x-www-form-urlencoded',
        'content' => http_build_query(['key' => $key, 'domain' => INSTALL_DOMAIN, 'action' => 'verify']),
        'timeout' => 8
    ]]);
    $resp = @file_get_contents(EKLESIA_VERIFY, false, $ctx);
    if (!$resp) {
        // Se não consegue contactar o servidor (offline), aceitar por 3 dias extra
        $lic = json_decode(file_get_contents(LICENSE_FILE), true);
        $lastVerify = $lic['last_verify'] ?? 0;
        return (time() - $lastVerify) < 86400 * 10;
    }
    $data = json_decode($resp, true);
    if (!empty($data['valid'])) {
        // Actualizar timestamp de verificação
        $lic = json_decode(file_get_contents(LICENSE_FILE), true);
        $lic['last_verify'] = time();
        file_put_contents(LICENSE_FILE, json_encode($lic, JSON_PRETTY_PRINT));
        return true;
    }
    return false;
}

function activateKey(string $key): array {
    $ctx = stream_context_create(['http' => [
        'method'  => 'POST',
        'header'  => 'Content-Type: application/x-www-form-urlencoded',
        'content' => http_build_query(['key' => $key, 'domain' => INSTALL_DOMAIN, 'action' => 'activate']),
        'timeout' => 12
    ]]);
    $resp = @file_get_contents(EKLESIA_ACTIVATE, false, $ctx);
    if (!$resp) {
        return ['ok' => false, 'msg' => 'Não foi possível contactar o servidor de licenças. Verifique a ligação à internet.'];
    }
    $data = json_decode($resp, true);
    if (!empty($data['ok'])) {
        file_put_contents(LICENSE_FILE, json_encode([
            'key'         => $key,
            'domain'      => INSTALL_DOMAIN,
            'activated'   => true,
            'activated_at'=> date('Y-m-d H:i:s'),
            'last_verify' => time()
        ], JSON_PRETTY_PRINT));
        return ['ok' => true, 'msg' => $data['msg'] ?? 'Activado com sucesso.'];
    }
    return ['ok' => false, 'msg' => $data['msg'] ?? 'Chave inválida ou já utilizada noutro domínio.'];
}

// ─── INICIALIZAR SISTEMA ─────────────────────────────────────
if (!file_exists(SYSTEM_FILE)) {
    $system = [
        'church_name' => '',
        'admin' => [
            'email'      => '',
            'password'   => '',
            'name'       => '',
            'created_at' => ''
        ],
        'installed'   => false,
        'installed_at'=> ''
    ];
    file_put_contents(SYSTEM_FILE, json_encode($system, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

// ─── FUNÇÕES DE SESSÃO ────────────────────────────────────────
function currentUser(): ?array {
    return $_SESSION['eklesia_user'] ?? null;
}

function isLoggedIn(): bool {
    return isset($_SESSION['eklesia_user']) && !empty($_SESSION['eklesia_user']['role']);
}
