<?php
// ─── Leitura / escrita JSON ───────────────────────────────────────────────
function readJSON(string $path): array {
    if (!file_exists($path)) return [];
    $content = file_get_contents($path);
    return json_decode($content, true) ?? [];
}

function writeJSON(string $path, $data): bool {
    $dir = dirname($path);
    if (!is_dir($dir)) mkdir($dir, 0755, true);
    return file_put_contents($path, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)) !== false;
}

// ─── UUID v4 simples ──────────────────────────────────────────────────────
function uuid(): string {
    return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
        mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff),
        mt_rand(0, 0x0fff) | 0x4000,
        mt_rand(0, 0x3fff) | 0x8000,
        mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
    );
}

// ─── Resposta JSON para API ───────────────────────────────────────────────
function jsonResponse(bool $success, $data = null, string $message = '', int $code = 200): void {
    http_response_code($code);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode([
        'success' => $success,
        'message' => $message,
        'data'    => $data
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

// ─── Sessão do utilizador ─────────────────────────────────────────────────
function currentUser(): ?array {
    return $_SESSION['eklesia_user'] ?? null;
}

function requireAuth(): void {
    $user = currentUser();
    if (!$user) {
        if (isAjax()) {
            jsonResponse(false, null, 'Sessão expirada. Faça login novamente.', 401);
        }
        header('Location: /index.php?expired=1');
        exit;
    }
}

function requireRole(array $roles): void {
    requireAuth();
    $user = currentUser();
    if (!in_array($user['role'], $roles)) {
        if (isAjax()) {
            jsonResponse(false, null, 'Acesso não autorizado.', 403);
        }
        header('Location: /dashboard.php');
        exit;
    }
}

function isAjax(): bool {
    return !empty($_SERVER['HTTP_X_REQUESTED_WITH']) &&
           strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';
}

// ─── Caminhos de dados da igreja ─────────────────────────────────────────
function churchPath(string $churchId, string $file = ''): string {
    $base = CHURCHES_DIR . '/' . $churchId;
    if (!is_dir($base)) mkdir($base, 0755, true);
    return $file ? $base . '/' . $file : $base;
}

function getChurchInfo(string $churchId): ?array {
    $info = readJSON(churchPath($churchId, 'info.json'));
    return empty($info) ? null : $info;
}

// ─── Sanitizar input ─────────────────────────────────────────────────────
function sanitize($value) {
    if (is_array($value)) return array_map('sanitize', $value);
    return htmlspecialchars(strip_tags(trim((string)$value)), ENT_QUOTES, 'UTF-8');
}

function post(string $key, $default = ''): string {
    return sanitize($_POST[$key] ?? $default);
}

function postRaw(string $key, $default = ''): string {
    return trim($_POST[$key] ?? $default);
}

function getInput(): array {
    $raw = file_get_contents('php://input');
    return json_decode($raw, true) ?? [];
}

// ─── Formatar data ────────────────────────────────────────────────────────
function fmtDate(?string $date): string {
    if (!$date) return '—';
    try {
        return (new DateTime($date))->format('d/m/Y');
    } catch (Exception $e) {
        return $date;
    }
}
