<?php
// ═══════════════════════════════════════════════════════════════
//  EKLESIA STANDALONE — layout.php
// ═══════════════════════════════════════════════════════════════

function getSystemInfo(): array {
    if (!file_exists(SYSTEM_FILE)) return [];
    return json_decode(file_get_contents(SYSTEM_FILE), true) ?? [];
}

function getChurchDisplayName(): string {
    $sys = getSystemInfo();
    return $sys['church_name'] ?? 'Eklesia';
}

function renderHead(string $title = 'Eklesia'): void {
    $church = getChurchDisplayName();
    echo "<!DOCTYPE html>
<html lang=\"" . (getLang()==='en'?'en':'pt') . "\">
<head>
<meta charset=\"UTF-8\">
<meta name=\"viewport\" content=\"width=device-width,initial-scale=1\">
<meta name=\"robots\" content=\"noindex, nofollow\">
<title>" . htmlspecialchars($title) . " — " . htmlspecialchars($church) . "</title>
<link rel=\"preconnect\" href=\"https://fonts.googleapis.com\">
<link href=\"https://fonts.googleapis.com/css2?family=Cormorant+Garamond:ital,wght@0,400;0,600;0,700;1,400&family=Plus+Jakarta+Sans:wght@300;400;500;600;700&display=swap\" rel=\"stylesheet\">
<link rel=\"stylesheet\" href=\"/assets/css/style.css\">
</head>";
}

function getBaseUrlStandalone(): string {
    // Usar root-relative — mais seguro e compatível com qualquer instalação
    return '';
}

function renderSidebar(string $role, ?array $sys = null): void {
    $lang   = getLang();
    $base   = getBaseUrlStandalone();
    $church = getChurchDisplayName();
    $user   = currentUser();
    $name   = $user['name'] ?? '';
    $roleLabel = match($role) {
        'admin'        => ($lang==='en' ? 'Administrator' : 'Administrador'),
        'local_user'   => ($lang==='en' ? 'Local Manager' : 'Gestor Local'),
        'finance_user' => ($lang==='en' ? 'Finance Manager' : 'Gestor Financeiro'),
        default        => $role
    };

    $otherLang  = $lang === 'en' ? 'pt' : 'en';
    $langLabel  = $lang === 'en' ? '🌐 Português' : '🌐 English';

    $nav = '';
    if ($role === 'admin') {
        $nav .= navItem($base.'/church/', '🏠', $lang==='en'?'Dashboard':'Painel', 'church');
        $nav .= navSection($lang==='en'?'Members':'Membros');
        $nav .= navItem($base.'/church/members.php', '👥', $lang==='en'?'Members':'Membros', 'members');
        $nav .= navItem($base.'/church/leaders.php', '👑', $lang==='en'?'Leaders':'Líderes', 'leaders');
        $nav .= navItem($base.'/church/ministries.php', '⛪', $lang==='en'?'Ministries':'Ministérios', 'ministries');
        $nav .= navSection($lang==='en'?'Management':'Gestão');
        $nav .= navItem($base.'/church/setup.php', '⚙️', $lang==='en'?'Settings':'Configurações', 'setup');
        $nav .= navItem($base.'/church/users.php', '👤', $lang==='en'?'Users':'Utilizadores', 'users');
        $nav .= navItem($base.'/church/reports.php', '📊', $lang==='en'?'Reports':'Relatórios', 'reports');
        $nav .= navSection($lang==='en'?'Finance':'Finanças');
        $nav .= navItem($base.'/finance/', '💰', $lang==='en'?'Finances':'Finanças', 'finance');
    } elseif ($role === 'local_user') {
        $nav .= navItem($base.'/local/', '🏠', $lang==='en'?'Dashboard':'Painel', 'local');
        $nav .= navItem($base.'/local/members.php', '👥', $lang==='en'?'Members':'Membros', 'members');
        $nav .= navItem($base.'/local/reports.php', '📊', $lang==='en'?'Reports':'Relatórios', 'reports');
    } elseif ($role === 'finance_user') {
        $nav .= navItem($base.'/finance/', '💰', $lang==='en'?'Finances':'Finanças', 'finance');
        $nav .= navItem($base.'/finance/categories.php', '🗂️', $lang==='en'?'Categories':'Categorias', 'categories');
        $nav .= navItem($base.'/finance/reports.php', '📊', $lang==='en'?'Reports':'Relatórios', 'reports');
    }

    $initials = strtoupper(mb_substr($name, 0, 1));

    echo "
<div class='app-layout'>
<nav class='sidebar' id='sidebar'>
    <div class='sidebar-brand'>
        <div class='sidebar-mini-cross'>✝</div>
        <div>
            <div class='sidebar-logo-text'>" . htmlspecialchars(mb_strtoupper(mb_substr($church,0,8))) . "</div>
            <div class='sidebar-church-name'>" . htmlspecialchars($church) . "</div>
        </div>
    </div>
    <nav class='sidebar-nav'>{$nav}</nav>
    <div class='sidebar-footer'>
        <a href='?lang={$otherLang}' class='sidebar-lang-btn'>{$langLabel}</a>
        <div class='sidebar-user'>
            <div class='user-avatar'>{$initials}</div>
            <div class='user-info'>
                <div class='user-name'>" . htmlspecialchars($name) . "</div>
                <div class='user-role'>{$roleLabel}</div>
            </div>
            <a href='{$base}/logout.php' title='" . ($lang==='en'?'Logout':'Sair') . "' style='color:rgba(255,255,255,0.3);font-size:1.1rem;text-decoration:none;margin-left:auto'>⇥</a>
        </div>
    </div>
</nav>
<div class='sidebar-overlay' id='sidebarOverlay' onclick='toggleSidebar()'></div>";
}

function navSection(string $label): string {
    return "<div class='nav-section-title'>{$label}</div>";
}

function navItem(string $href, string $icon, string $label, string $key): string {
    $current = basename($_SERVER['PHP_SELF'], '.php');
    $path    = parse_url($href, PHP_URL_PATH) ?? '';
    $active  = (strpos($path, $key) !== false) ? ' active' : '';
    return "<a href='{$href}' class='nav-item{$active}'><span class='nav-icon'>{$icon}</span>{$label}</a>";
}

function renderFlash(): void {
    $f = getFlash();
    if (!$f) return;
    $type = $f['type'] === 'success' ? 'alert-success' : ($f['type'] === 'danger' ? 'alert-danger' : 'alert-info');
    $icon = $f['type'] === 'success' ? '✅' : ($f['type'] === 'danger' ? '❌' : 'ℹ️');
    echo "<div class='alert {$type}'>{$icon} " . $f['msg'] . "</div>";
}

function renderSuggestionBanner(): void {
    $lang = getLang();
    $msg  = $lang==='en'
        ? 'Does your church have an idea to improve this system? Want to suggest a feature or report an issue?'
        : 'A sua denominação tem uma ideia para melhorar o sistema? Quer sugerir uma funcionalidade ou reportar um problema?';
    $cta  = $lang==='en' ? '✉️ Send suggestion →' : '✉️ Enviar sugestão →';
    $subj = $lang==='en' ? 'Eklesia Standalone — Suggestion' : 'Eklesia Standalone — Sugestão';
    echo "
<div class='suggestion-banner' id='suggBanner'>
    <div class='suggestion-inner'>
        <span class='suggestion-icon'>💡</span>
        <p class='suggestion-text'>{$msg}</p>
        <a href='mailto:" . CONTACT_EMAIL . "?subject=" . urlencode($subj) . "' class='suggestion-btn'>{$cta}</a>
        <button class='suggestion-close' onclick=\"document.getElementById('suggBanner').style.display='none';localStorage.setItem('ssugg','1')\" title='Fechar'>✕</button>
    </div>
</div>
<script>if(localStorage.getItem('ssugg')==='1'){var b=document.getElementById('suggBanner');if(b)b.style.display='none';}</script>";
}

function renderFooter(): void {
    $church = htmlspecialchars(getChurchDisplayName());
    $year   = date('Y');
    echo "
<footer class='standalone-footer'>
    <strong>{$church}</strong> &copy; {$year}
    <span class='footer-sep'>·</span>
    <a href='https://eklesia.asaf.fyi' target='_blank' rel='noopener'>um produto da eklesia.asaf.fyi</a>
</footer>";
}

function renderJS(): void {
    echo "
<script>
function toggleSidebar(){
    document.getElementById('sidebar').classList.toggle('open');
    document.getElementById('sidebarOverlay').classList.toggle('active');
}
// auto-dismiss alerts
document.querySelectorAll('.alert').forEach(function(a){
    setTimeout(function(){ a.style.opacity='0'; a.style.transition='opacity .5s'; setTimeout(function(){ a.remove(); },500); }, 5000);
});
// tabs
document.querySelectorAll('.tab-btn').forEach(function(b){
    b.addEventListener('click',function(){
        var t=this.dataset.tab;
        this.closest('.tabs').querySelectorAll('.tab-btn').forEach(function(x){ x.classList.remove('active'); });
        this.classList.add('active');
        document.querySelectorAll('.tab-content').forEach(function(c){ c.classList.remove('active'); });
        var tc=document.getElementById(t);
        if(tc) tc.classList.add('active');
    });
});
// modal
document.querySelectorAll('[data-modal]').forEach(function(b){
    b.addEventListener('click',function(){
        var m=document.getElementById(this.dataset.modal);
        if(m) m.classList.add('active');
    });
});
document.querySelectorAll('.modal-close').forEach(function(b){
    b.addEventListener('click',function(){
        this.closest('.modal-overlay').classList.remove('active');
    });
});
document.querySelectorAll('.modal-overlay').forEach(function(o){
    o.addEventListener('click',function(e){ if(e.target===this) this.classList.remove('active'); });
});
</script>
</body></html>";
}
